"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.vendorController = exports.VendorController = void 0;
const service_1 = require("./service");
/**
 * Vendor Controller - Handles vendor-related operations
 */
class VendorController {
    /**
     * Get current vendor profile
     * GET /api/vendors/me
     */
    async getMyProfile(req, res) {
        try {
            if (!req.user) {
                return res.status(401).json({
                    success: false,
                    error: 'Authentication required'
                });
            }
            const { data: vendor, error } = await service_1.vendorService.getVendorByUserId(req.user.id);
            if (error) {
                console.error('Get vendor profile error:', error);
                return res.status(500).json({
                    success: false,
                    error: 'Failed to retrieve vendor profile'
                });
            }
            if (!vendor) {
                return res.status(404).json({
                    success: false,
                    error: 'Vendor profile not found'
                });
            }
            res.status(200).json({
                success: true,
                data: vendor
            });
            return;
        }
        catch (error) {
            console.error('Get my profile error:', error);
            res.status(500).json({
                success: false,
                error: 'Failed to retrieve profile'
            });
            return;
        }
    }
    /**
     * Complete vendor onboarding
     * POST /api/vendors/onboarding
     */
    async completeOnboarding(req, res) {
        try {
            if (!req.user) {
                return res.status(401).json({
                    success: false,
                    error: 'Authentication required'
                });
            }
            const onboardingData = req.body;
            // Validate required fields
            if (!onboardingData.business_name) {
                return res.status(400).json({
                    success: false,
                    error: 'Business name is required'
                });
            }
            // Get current vendor
            const { data: currentVendor, error: getError } = await service_1.vendorService.getVendorByUserId(req.user.id);
            if (getError || !currentVendor) {
                return res.status(404).json({
                    success: false,
                    error: 'Vendor profile not found'
                });
            }
            // Update onboarding information
            const { data: updatedVendor, error: updateError } = await service_1.vendorService.updateOnboarding(currentVendor.id, onboardingData);
            if (updateError) {
                console.error('Update onboarding error:', updateError);
                return res.status(500).json({
                    success: false,
                    error: 'Failed to update onboarding information'
                });
            }
            res.status(200).json({
                success: true,
                data: updatedVendor,
                message: 'Onboarding completed successfully'
            });
            return;
        }
        catch (error) {
            console.error('Complete onboarding error:', error);
            res.status(500).json({
                success: false,
                error: 'Failed to complete onboarding'
            });
            return;
        }
    }
    /**
     * Update vendor profile
     * PUT /api/vendors/profile
     */
    async updateProfile(req, res) {
        try {
            if (!req.user) {
                return res.status(401).json({
                    success: false,
                    error: 'Authentication required'
                });
            }
            const updateData = req.body;
            // Get current vendor
            const { data: currentVendor, error: getError } = await service_1.vendorService.getVendorByUserId(req.user.id);
            if (getError || !currentVendor) {
                return res.status(404).json({
                    success: false,
                    error: 'Vendor profile not found'
                });
            }
            // Update vendor profile
            const { data: updatedVendor, error: updateError } = await service_1.vendorService.updateVendor(currentVendor.id, updateData);
            if (updateError) {
                console.error('Update profile error:', updateError);
                return res.status(500).json({
                    success: false,
                    error: 'Failed to update profile'
                });
            }
            res.status(200).json({
                success: true,
                data: updatedVendor,
                message: 'Profile updated successfully'
            });
            return;
        }
        catch (error) {
            console.error('Update profile error:', error);
            res.status(500).json({
                success: false,
                error: 'Failed to update profile'
            });
            return;
        }
    }
    /**
     * Get vendor by ID (admin only)
     * GET /api/vendors/:id
     */
    async getVendorById(req, res) {
        try {
            const { id } = req.params;
            const { data: vendor, error } = await service_1.vendorService.getVendorById(id);
            if (error) {
                console.error('Get vendor by ID error:', error);
                return res.status(500).json({
                    success: false,
                    error: 'Failed to retrieve vendor'
                });
            }
            if (!vendor) {
                return res.status(404).json({
                    success: false,
                    error: 'Vendor not found'
                });
            }
            res.status(200).json({
                success: true,
                data: vendor
            });
            return;
        }
        catch (error) {
            console.error('Get vendor by ID error:', error);
            res.status(500).json({
                success: false,
                error: 'Failed to retrieve vendor'
            });
            return;
        }
    }
    /**
     * Get all vendors (admin only)
     * GET /api/vendors
     */
    async getAllVendors(req, res) {
        try {
            const { data: vendors, error } = await service_1.vendorService.getAllVendors();
            if (error) {
                console.error('Get all vendors error:', error);
                return res.status(500).json({
                    success: false,
                    error: 'Failed to retrieve vendors'
                });
            }
            res.status(200).json({
                success: true,
                data: vendors
            });
            return;
        }
        catch (error) {
            console.error('Get all vendors error:', error);
            res.status(500).json({
                success: false,
                error: 'Failed to retrieve vendors'
            });
            return;
        }
    }
}
exports.VendorController = VendorController;
exports.vendorController = new VendorController();
//# sourceMappingURL=controller.js.map